#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
quantile4pdf <- function(X,
                         q=c(0.025,0.25,0.50,0.75,0.975),
                         from=min(X),to=max(X))
#TITLE returns the quantiles of the densities estimated
# from a two-dimensional sample
#DESCRIPTION
#   Let be \code{X} a matrix as defined in \code{make8cdf}.
#   This function returns a matrix of the desired quantiles
#   indicated in \code{q} of its densities by columns.
#DETAILS
# Just an empirical trial to see... Already as drawback, it can
# be said that the drawn median IS NOT the density of a probability
# distribution, in the sense that it does not sum to one.
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{X} <<Matrix of the simulated values.>>
#[INPUTS]
#{q} <<Vector of the desired quantiles.>>
#{from} <<lower bound for the density.>>
#{to}   <<upper bound for the density.>>
#VALUE
# A matrix of dimension \code{512 x length(q)+1}. Its
# columns returns the sorted values of the estimated
# densities by the function \code{density}). The last column
# is the common associated values of \code{density()$x}.
#EXAMPLE
# quantile4pdf(matrix(runif(1000),100));
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_22
#REVISED 11_09_23
#--------------------------------------------
{
  # constants
  nv <- nrow(X); nu <- ncol(X); nq <- length(q);
  nbd <- 512;
  # computing the columns by columns densities
  dd <- matrix(NA,nbd,nv);
  dx <- density(X[,1],n=nbd,from=from,to=to)$x;
  for (u in bc(nu)) {
    dd[,u] <- density(X[,u],n=nbd,from=from,to=to)$y;
  }
  # computing the quantiles
  resu <- matrix(NA,nbd,nq+1);
  for (i in bc(nbd)) {
    resu[i,-(nq+1)] <- quantile(dd[i,],q,na.rm=TRUE);
  }
  resu[,nq+1] <- dx;
  # returning
  resu;
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 
#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
plot8smoothedpdf <- function(X,from=min(X),to=max(X),
                             cov=c(0.50,0.95),col=c("grey","lightgrey"))
#TITLE plots a cdf with median and color coverage according to precised levels
#DESCRIPTION
#   Let be \code{X} a matrix containing simulations as described
#   in \code{quantile4cdf}. The set of implicitely included densities
#   associated to each column is displayed with coloured zones.
#DETAILS
#   The plot is supposed opened before calling. The used zone will
#   comprised into the rectangle \code{c(0,1) x range(X)}.
#   This is an experimental graphics: just to see and seek new ideas.
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{X} <<Matrix of the simulated values.>>
#[INPUTS]
#{from} <<lower bound for the density.>>
#{to}   <<upper bound for the density.>>
#{cov} <<Vector of the desired coverage probabilities.>>
#{col} <<Vector of the associated colours.>>
#VALUE
# Nothing but a plot is issued.
#EXAMPLE
# plot(... to be fitted with the example
# plot8smoothedpdf(matrix(runif(1000),100));
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_23
#REVISED 11_09_23
#--------------------------------------------
{
  # constants
  nbz <- length(cov);
  # computing the necessary quantiles for the coverage
  qqq <- sort(c(1-(1-cov)/2,(1-cov)/2,0.5));
  # computing the quantile densities
  X <- quantile4pdf(X,q=qqq,from=from,to=to);
  # coloring the different zones
  for (i in bc(nbz)) {
    c1 <- i; c2 <- (2*nbz+1)+1-i;
    xx <- c(X[,2*nbz+2],X[nrow(X):1,2*nbz+2]);
    yy <- c(X[,c1],X[nrow(X):1,c2]);
    polygon(xx,yy,border=NA,col=col[nbz+1-i]);
  }
  # adding the median
  lines(X[,2*nbz+2],X[,nbz+1],lwd=3);
  # returning
  invisible();
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 
#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
plot8rawpdf <- function(X,from=min(X),to=max(X),lty=1,lwd=1)
#TITLE plots the direct pdf of a two-way simulated matrix
#DESCRIPTION
#   Let be \code{X} a matrix containing simulations as described
#   in \code{make4cdf}. Each column of it is displayed as a density.
#DETAILS
#   The plot is supposed opened before calling. The used zone will
#   comprised into the rectangle \code{range(X) x max(density(X[,*])$y)}.
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{X} <<Matrix of the simulated values.>>
#[INPUTS]
#{from} <<lower bound for the density.>>
#{to}   <<upper bound for the density.>>
#{lty} <<type of the lines to draw.
#        can be a vector of length \code{ncol(X)}
#        to associate different types of lines at
#        each column.>>
#{lwd} <<width of the lines to draw; as \code{lty}.>>
#VALUE
# Nothing but a plot is issued.
#EXAMPLE
# plot8rawcdf(matrix(runif(1000),100));
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_23
#REVISED 11_09_23
#--------------------------------------------
{
  # decoding the arguments
  if (length(lty)==1) { lty <- rep(lty,ncol(X));}
  if (length(lwd)==1) { lwd <- rep(lwd,ncol(X));}
  # drawing a density a column
  for (j in bc(ncol(X))) {
    lines(density(X[,j],from=from,to=to),lty=lty[j],lwd=lwd[j]);
  }
  # returning
  invisible();
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
