#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
make8cdf <- function(X)
#TITLE returns the empirical cdf from a mc2d simulation
#DESCRIPTION
#   Let be \code{X} a matrix containing in each of its columns
#   an i.i.d. sample of distributions. These distributions are
#   themselves an i.i.d. sample of a random distribution.
#   This is the framework of 2-dimensional simulation. Usually
#   the columns are associated to the uncertainty and the rows
#   to the variability nested within uncertainty.
#   This function sort values for each column producing easy to
#   tackle the simulations as cdf.
#DETAILS
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{X} <<Matrix of the simulated values.>>
#[INPUTS]
#VALUE
# A matrix with the same dimension of \code{X}; the
# columns being sorted.
#EXAMPLE
# make4cdf(matrix(runif(1000),100));
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_23
#REVISED 11_09_23
#--------------------------------------------
{
  # constants
  nu <- ncol(X);
  # performing the necessary sorts
  for (u in bc(nu)) { X[,u] <- sort(X[,u]);}
  # returning
  X;
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 
#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
quantile4cdf <- function(X,q=c(0.025,0.25,0.50,0.75,0.975))
#TITLE returns the quantiles of the cdf estimated from a two-dimensional sample
#DESCRIPTION
#   Let be \code{X} a matrix as defined in \code{make8cdf}.
#   This function returns a matrix of the desired quantiles
#   indicated in \code{q} of its cdf by columns.
#DETAILS
# The justification of such an algorithm is based on the fact that the
# quantiles of a set of cumulative distribution functions [cdf] is
# identical looking at them from x or y viewpoints. More explicitely,
# when taking the intersections of a bundle of cdfs with a varying vertical
# line, or with a varying horizontal line (between 0 and 1), one
# obtains the same set of points (if the quantile is exact with the
# number of cdfs). Of course with in inversion to get the same sense.
# 
# This can be very simply proved by noticing that the 0 quantile
# (minimum) with respect to \code{x} is the 1 quantile (maximum) with
# respect to \code{y}.
# 
# From this property, the used algorithm is deduced when the empirical
# cdf are based on the same number of simulations (here
# \code{nrow(X)}). Just sorting them within each row.
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{X} <<Matrix of the simulated values.>>
#[INPUTS]
#{q} <<Vector of the desired quantiles.>>
#VALUE
# A matrix of dimension \code{nrow(X) x length(q)}. Its
# columns returns the sorted values of the quantile cdf.
#EXAMPLE
# quantile4cdf(make4cdf(matrix(runif(1000),100)));
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_22
#REVISED 11_09_23
#--------------------------------------------
{
  # constants
  nv <- nrow(X); nq <- length(q);
  # performing the necessary sorts
  X <- make8cdf(X);
  for (v in bc(nv)) { X[v,] <- sort(X[v,]);}
  # computing the quantiles
  resu <- matrix(NA,nv,nq);
  for (i in bc(nv)) {
    resu[i,] <- quantile(X[i,],q,na.rm=TRUE);
  }
  # returning
  resu;
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 
#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
plot8smoothedcdf <- function(X,cov=c(0.50,0.95),col=c("grey","lightgrey"))
#TITLE plots a cdf with median and color coverage according to precised levels
#DESCRIPTION
#   Let be \code{X} a matrix containing simulations as described
#   in \code{quantile4cdf}. The set of implicitely included cdf
#   associated to each column is displayed with coloured zones.
#DETAILS
#   The plot is supposed opened before calling. The used zone will
#   comprised into the rectangle \code{range(X) x c(0,1)}.
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{X} <<Matrix of the simulated values.>>
#[INPUTS]
#{cov} <<Vector of the desired coverage probabilities.>>
#{col} <<Vector of the associated colours.>>
#VALUE
# Nothing but a plot is issued.
#EXAMPLE
# plot8cdf(matrix(runif(1000),100));
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_22
#REVISED 11_09_22
#--------------------------------------------
{
  # constants
  nbz <- length(cov);
  # computing the necessary quantiles for the coverage
  qqq <- sort(c(1-(1-cov)/2,(1-cov)/2,0.5));
  # computing the quantile cdf
  X <- quantile4cdf(X,qqq);
  # coloring the different zones
  for (i in bc(nbz)) {
    c1 <- i; c2 <- (2*nbz+1)+1-i;
    xx <- c(X[,c1],X[nrow(X):1,c2]);
    yy <- c(seq(0,1,length=nrow(X)),seq(1,0,length=nrow(X)));
    polygon(xx,yy,border=NA,col=col[nbz+1-i]);
  }
  # adding the median
  lines(X[,nbz+1],seq(0,1,length=nrow(X)),lwd=3);
  # returning
  invisible();
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 
#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
plot8rawcdf <- function(X,lty=1,lwd=1)
#TITLE plots the direct cdf of a two-way simulated matrix?
#DESCRIPTION
#   Let be \code{X} a matrix containing simulations as described
#   in \code{make4cdf}. Each column of it is displayed as a cdf.
#DETAILS
#   The plot is supposed opened before calling. The used zone will
#   comprised into the rectangle \code{range(X) x c(0,1)}.
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{X} <<Matrix of the simulated values.>>
#[INPUTS]
#{lty} <<type of the lines to draw.
#        can be a vector of length \code{ncol(X)}
#        to associate different types of lines at
#        each column.>>
#{lwd} <<width of the lines to draw; as \code{lty}.>>
#VALUE
# Nothing but a plot is issued.
#EXAMPLE
# plot8rawcdf(matrix(runif(1000),100));
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_22
#REVISED 11_09_22
#--------------------------------------------
{
  # decoding the arguments
  if (length(lty)==1) { lty <- rep(lty,ncol(X));}
  if (length(lwd)==1) { lwd <- rep(lwd,ncol(X));}
  # making the cdf
  X <- make8cdf(X);
  # drawing a cdf a column
  for (j in bc(ncol(X))) {
    lines(X[,j],seq(0,1,length=nrow(X)),lty=lty[j],lwd=lwd[j]);
  }
  # returning
  invisible();
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
