#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
margibibe <- function(n,a,b)
#TITLE returns the marginal probabilities of a binomial * beta
#DESCRIPTION
#   returns the vector of the \code{n+1} probabilities
# of a discrete random variable defined as a binomial of
# size \code{n} with a parameter \code{p} following
# a beta distribution with parameters \code{a} and \code{b}.
#DETAILS
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{n} <<Size of the binomial.>>
#{a} <<First parameter of the beta.>>
#{b} <<Second parameter of the beta.>>
#[INPUTS]
#VALUE
# a vector with \code{n+1} probabilities summing one,
# associated to the values \code{0:n}.
#EXAMPLE
# margibibe(10,2,5);
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_20
#REVISED 11_09_20
#--------------------------------------------
{
  # checking
  check4tyle(n, "integer",1, c(0,Inf),message = "'n' is not valid");
  check4tyle(a, "numeric",1, c(0,Inf),message = "'a' is not valid");
  check4tyle(b, "numeric",1, c(0,Inf),message = "'b' is not valid");
  # protection
  if(n+a+b > 120) {
    erreur(c(n,a,b),"Arguments with too high values");
  }
  # special case
  if (n==0) { return(1);}
  # the general constant
  kk <- gamma(a+b)/gamma(a)/gamma(b)*gamma(n+1)/gamma(n+a+b);
  # the varying part
  vv <- 0:n;
  kv <- gamma(a+vv)/gamma(1+vv)*gamma(b+n-vv)/gamma(1+n-vv);
  # returning
  kk*kv;
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
draw8histo <- function(ou,
                      hist,
                      echelle=1,
                      orientation="N",
                      grafo=vector("list",0))
#TITLE draws an histogram within a plot
#DESCRIPTION
#   draws the density of any discrete distribution
#   with rectangles (possibly colored) in any of the
#   four cardinal directions
#DETAILS
# The plot is supposed already open. 
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{ou} <<Defines the rectangle within which the histogram
#       has to be drawn. Must be a vector of length 4 with
#       \code{c(xA,xB,yA,yB)} defining two opposite corners
#       of the rectangle.
#       Its extreme values correspond to
#       \code{range(hist$n) + 0.5*c(-1,1)} and to
#       \code{c(0,max(hist$f))}.  >>
#{hist} <<List with two components of equal lengths: \code{$n}
#         providing the integer values and \code{$f} providing
#         the associated frequencies.>>
#[INPUTS]
#{echelle} <<Multiplier coefficient to be applied before
#            drawing the bars once they have fitted to the maximum
#            variation within the given box.>>
#{orientation} <<Indicates the way the histogram must be drawn
#                within the rectangle. Possible values are \code{"N"} for
#                towards North, \code{"W"} towards West,...>>
#{grafo} <<List with possible components giving the style of the
#           bars for each value of the histogram. When they don't
#           exist or are null, default values are used. The components
#           are: \code{$lty} for the style of the line (0 means no line),
#           \code{$lwd} for the width of the line (0 means no line),
#           \code{$wid} width of the bar (1 means that the consecutive
#           bars will be in contact),
#           \code{$col} the color to fill the bar (0 no filling).>>
#VALUE
# Nothing, some histogram is added to the current plot.
#EXAMPLE
# plot(0,0,xlim=c(-1,6),ylim=c(-0.1,1.1),type="n");
# draw8histo(ou=c(0,5,0,1),hist=list(n=1:5,f=1:5));
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_20
#REVISED 11_09_20
#--------------------------------------------
{
  # checking
  check4tyle(ou, "numeric", 4, message = "'ou' must be a numeric(4)");
  check4tyle(hist,"list",2, message = "'hist' must be a list(2)");
  check4tyle(orientation,"character",1,c("N","W","S","E"),
             message = "'orientation' is not valid");
  # decoding
  nb <- length(hist$n);
  xl <- range(ou[1:2]);
  yl <- range(ou[3:4]);
  if (is.null(grafo$lty)) { grafo$lty <- rep(1,nb);}
  if (is.null(grafo$lwd)) { grafo$lwd <- rep(1,nb);}
  if (is.null(grafo$wid)) { grafo$wid <- rep(1,nb);}
  if (is.null(grafo$col)) { grafo$col <- rep(NA,nb);}
  # just drawing the asked bars
  for (bar in bc(nb)) {
    # getting the assocated rectangle
    if (orientation=="N") {
      # abscissa
      xx <- rep(( 0.5 + (hist$n[bar]-min(hist$n))),4);
      xx <- xx + grafo$wid[bar]*c(-1,1,1,-1)/2;
      xx <- xl[1] + xx * diff(range(xl)) / (diff(range(hist$n))+1);
      # ordinates
      yy <- rep(yl[1],4);
      yy[3:4] <- yy[3:4] +
                 hist$f[bar]/max(hist$f) * rep(diff(yl),2) * echelle;
    }
    if (orientation=="S") {
      # abscissa
      xx <- rep(( 0.5 + (hist$n[bar]-min(hist$n))),4);
      xx <- xx + grafo$wid[bar]*c(-1,1,1,-1)/2;
      xx <- xl[1] + xx * diff(range(xl)) / (diff(range(hist$n))+1);
      # ordinates
      yy <- rep(yl[2],4);
      yy[3:4] <- yy[3:4] -
                 hist$f[bar]/max(hist$f) * rep(diff(yl),2) * echelle;
    }
    if (orientation %in% c("E","W")) {
      pause(paste("Sorry, this orientation (",orientation,") is not yet implemented"));
    }
    # drawing it
    polygon(xx,yy,lty=grafo$lty[bar],
            lwd=grafo$lwd[bar],col=grafo$col[bar]
           );
  }
  # returning
  invisible();
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
draw8beta <- function(ou,
                     a,b,
                     pp=c(0.25,0.50,0.75),
                     ci=-1,
                     echelle=1,
                     orientation="E",
                     grafo=c("1","2","NA"),
                     ccol=c("red","green")
                    )
#TITLE draws a beta density within a plot
#DESCRIPTION
#   draws the density of a beta distribution
#   with parameters \code{a} and \code{b} in any of the
#   four cardinal directions
#DETAILS
# The plot is supposed already open. 
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{ou} <<Defines the rectangle within which the density
#       has to be drawn. Must be a vector of length 4 with
#       \code{c(xA,xB,yA,yB)} defining two opposite corners
#       of the rectangle.
#       Its extreme values correspond to
#       \code{0:1} and to \code{c(0,max(denisty))}.>>
#{a} <<First parameter of the beta distribution.>>
#{b} <<Second parameter of the beta distribution.>>
#[INPUTS]
#{pp} <<Some values of \code{p} to draw vertical segments on the 
#       density. When NULL nothing is drawn.>>
#{ci} <<Level for the credibility interval.>>
#{echelle} <<Multiplier coefficient to be applied before
#            drawing the density function once it has been fitted
#            to the maximum variation within the given box.>>
#{orientation} <<Indicates the way the histogram must be drawn
#                within the rectangle. Possible values are \code{"N"} for
#                towards North, \code{"W","S","E"}.>>
#{grafo} <<Character vector indicating the style of the draw.
#           First the style of the line, second the width of the line,
#           and third the color to fill the density. The color will not
#           be used if a confidence interval is asked for.>>
#{ccol} <<Colors to be used for the confidence interval.>>
#VALUE
# Nothing, some density is added to the current plot.
#EXAMPLE
# plot(0,0,xlim=c(-1,6),ylim=c(-0.1,1.1),type="n");
# draw8histo(ou=c(0,5,0,1),3,5);
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_20
#REVISED 11_09_21
#--------------------------------------------
{
  # constant
  nb <- 123;  
  # checking
  check4tyle(ou, "numeric", 4, message = "'ou' must be a numeric(4)");
  check4tyle(a,"numeric",1,c(0,Inf), message = "'a' must be a positive value");
  check4tyle(b,"numeric",1,c(0,Inf), message = "'b' must be a positive value");
  check4tyle(orientation,"character",1,c("N","W","S","E"),
             message = "'orientation' is not valid");
  check4tyle(grafo,"character",3,
             message = "'grafo' is not valid");
  # decoding
  if (is.null(pp)) { pp <- numeric(0);}
  xl <- range(ou[1:2]);
  yl <- range(ou[3:4]);
  lty <- as.numeric(grafo[1]);
  lwd <- as.numeric(grafo[2]);
  col <- grafo[3];
  if (col=="NA") { col <- NA;}
  vx <- seq(0,1,length=nb)[-c(1,nb)];
  credi <- ((ci > 0) & (ci < 1));
  vv <- dbeta(vx,a,b);
  if (credi) {
    cy <- qbeta(c((1-ci)/2,(1+ci)/2),a,b);
    quels <- (vx>cy[1])&(vx<cy[2]);
  }
  xp <- dbeta(pp,a,b)*echelle;
  mv <- max(vv);
  vv <- vv * echelle;
  # preparing the big polygon (and the small
  #  one for credibility interval)
  for (bar in bc(nb)) {
    # getting the assocated rectangle
    if (orientation=="W") {
      # abscissa
      xx <- c(0,vv,0);
      xx <- xl[2] - xx/mv*rep(diff(xl),nb);
      # ordinates
      yy <- c(0,vx,1);
      yy <- yl[1] + yy*rep(diff(yl),nb);
      if (credi) {
        xxp <- c(0,vv[quels],0);
        xxp <- xl[2] - xxp/mv*diff(xl);
        yyp <- c(cy[1],vx[quels],cy[2]);
        yyp <- yl[1] + yyp*diff(yl);
      }
    }
    if (orientation=="E") {
      # abscissa
      xx <- c(0,vv,0);
      xx <- xl[1] + xx/mv*rep(diff(xl),nb);
      # ordinates
      yy <- c(0,vx,1);
      yy <- yl[1] + yy*rep(diff(yl),nb);
      if (credi) {
        xxp <- c(0,vv[quels],0);
        xxp <- xl[1] + xxp/mv*diff(xl);
        yyp <- c(cy[1],vx[quels],cy[2]);
        yyp <- yl[1] + yyp*diff(yl);
      }
    }
    #
    if (orientation %in% c("N","S")) {
      pause(paste("Sorry, this orientation (",orientation,") is not yet implemented"));
    }
  }
  #
  if (credi) { col <- ccol[1];}
  # drawing it
  polygon(xx,yy,lty=lty,lwd=lwd,col=col
         );
  if (credi) {
    polygon(xxp,yyp,border=NA,col=ccol[2]
           );
  }
  # preparing and drawing the lines
  for (lig in bf(pp)) {
    # getting the assocated rectangle
    if (orientation=="W") {
      # abscissa
      xx <- xp[lig];
      xx <- xl[2] - xx/mv*diff(xl);
      xx <- c(xl[2],xx);
      # ordinates
      yy <- pp[lig];
      yy <- yl[1] + yy*diff(yl);
    }
    if (orientation=="E") {
      # abscissa
      xx <- xp[lig];
      xx <- xl[1] + xx/mv*diff(xl);
      xx <- c(xl[1],xx);
      # ordinates
      yy <- pp[lig];
      yy <- yl[1] + yy*diff(yl);
    }
    #
    lines(xx,rep(yy,2),lwd=4);
  }
  # returning
  invisible();
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
draw8bino <- function(ou,n,p,ci=0.95,echelle=1,ccol=c("red","orange","green"))
#TITLE draws a binomial
#DESCRIPTION
#   draws the density of a binomial distribution
#   with vertical bars with the help of \code{draw8histo}.
#DETAILS
# The plot is supposed already open.
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{ou} <<Rectangle to make the histogram (see \code{draw8histo}).>>
#{n} <<Size of the binomial>>
#{p} <<Probability of the binomial>>
#[INPUTS]
#{ci} <<Level for the credibility interval.>>
#{echelle} <<To be given to draw8histo. Multiplier coefficient to be applied before
#            drawing the bars once they have fitted to the maximum
#            variation within the given box.>>
#{ccol} <<Colors to be given to the different confidence zones.>>
#VALUE
# Nothing, some vertical bars are added to the current plot.
#EXAMPLE
# plot(0,0,xlim=c(-1,6),ylim=c(-0.1,1.1),type="n");
# draw8bino(ou=c(0,1,0,1),n=5,p=0.4);
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_19
#REVISED 13_06_11
#--------------------------------------------
{
  # preparing the call to draw8histo
  hist <- list(n=0:n,f=dbinom(0:n,n,p));
  #
  if ((ci > 0) & (ci < 1)) {
    colo <- rep(3,n+1);
    alpha <- (1 - ci) / 2;
    dd <- dbinom(0:n,n,p);
    cu1 <- cumsum(dd);
    cu2 <- cumsum(dd[(n+1):1])[(n+1):1];
    colo[(cu1 < alpha) | (cu2 < alpha)] <- 1;
    colo[sum(cu1 < alpha) + 1] <- 2;
    colo[n+1-sum(cu2 < alpha)] <- 2;
  } else {
    colo <- rep(NA,n+1);
  }
  form <- list(col=ccol[colo],wid=rep(1,n+1));
  hist <- list(n=0:n,f=dbinom(0:n,n,p));
  # drawing the histogram
  draw8histo(ou,hist,"N",form,echelle=echelle);
  # returning
  invisible();
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
draw8binos <- function(n,pp,ci=-1,
                       mti=NULL,sti=NULL,
                       xlimi=NULL,
                       main=NULL,sub=NULL,
                       xlab=NULL,ylab=NULL,
                       ccol=c("red","orange","green"),
                       ...)
#TITLE draws a binomial for different values of p
#DESCRIPTION
#   draws the density of binomial distributions
#   with vertical bars
#DETAILS
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{n} <<Size of the binomial>>
#{pp} <<Probabilities of the binomial>>
#[INPUTS]
#{ci} <<Value of the 'confidence interval'; see \code{draw8bino}.>>
#{mti} <<Main title for the plot.>>
#{sti} <<Sub title for the plot.>>
#{xlimi} <<imposing x limits for the plot.>>
#{main} << Main title of the plot.>>
#{sub} << Secondary title for the plot.>>
#{xlab} << x-axis label for the plot.>>
#{ylab} << y-axis label for the plot.>>
#{ccol} <<Colors to be given to the different confidence zones.>>
#{\dots} << to be given to the \code{plot} call.>>
#VALUE
# A complete plot is issued
#EXAMPLE
# draw8binos(5,c(0.4,0.5,0.7));
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_19
#REVISED 13_06_11
#--------------------------------------------
{
## separating constant
ysep <- 0.05;
## values to be considered
val <- 0:n;
## computing the range for abscissaes
xlim <- c(-1,1) + range(val);
## computing the position for each histogram
# height of each histo
hm <- rep(NA,length(pp));
for (ip in bf(pp)) {
  hm[ip] <- max(dbinom(val,n,pp[ip]));
}
# starting position of each histo
ou <- matrix(NA,length(pp),4);
for (ip in bf(pp)) {
  ou[ip,1:2] <- xlim;
  ou[ip,3:4] <- c(sum(hm[bc(ip-1)]) + ysep * sum(hm) * (ip-1),
                  sum(hm[bc(ip  )]) + ysep * sum(hm) * (ip-1));
}
## computing the range for ordinates
plus <- 1/5;
ylim <- c(-ysep-plus,1+length(pp)*ysep)*sum(hm);
## preparing the labels
lala <- unique(round(seq(0,n,le=7)));
## preparing the titles
if (is.null(mti)) { mti <- paste(length(pp),"binomial distribution(s)");}
if (is.null(sti)) { sti <- paste("size of",n);}
## opening the graph
if (is.null(xlimi)) { xlimi <- xlim;}
if (is.null(main)) { main <- mti;}
if (is.null(sub)) { sub <- sti;}
if (is.null(xlab)) { xlab <- "Binomial density";}
if (is.null(ylab)) { ylab <- paste("For p =",paste(pp,collapse=" / "));}
if (is.null(xlab)) { xlab <- "Binomial density";}
if (is.null(ylab)) { ylab <- paste("For p =",paste(pp,collapse=" / "));}
if (is.null(xlab)) { xlab <- "Binomial density";}
if (is.null(ylab)) { ylab <- paste("For p =",paste(pp,collapse=" / "));}
cat("xlim =",xlim,"ylim =",ylim,"\n");
plot(0,0,type="n",
     yaxt="n",xaxt="n",
     xlab=xlab,
     ylab=ylab,
     xlim=xlimi,ylim=ylim,
     main=main,sub=sub,
     ...
    );
## writing the labels
for (la in bf(lala)) {
  lal <- lala[la];
  lax <- 0.5 + ou[1,1] + lal * diff(ou[1,1:2]) / (diff(range(val))+1);
  text(lax,ylim[1]/2,lal,cex=1.5,pos=1);
}
## drawing each density
for (ip in bf(pp)) {
  draw8bino(ou[ip,],n,pp[ip],ci,ccol=ccol);
  abline(h=ou[ip,2],lwd=1);
}
## done
invisible();
}
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

#<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
draw8betabino <- function(n,a,b,
                         pp=c(0.25,0.50,0.75),
                         rr=NULL,
                         ci=-1,
                         type="joint",
                         grafo="complete",
                         mti=NULL,sti=NULL,
                         xlab=NULL,ylab=NULL,
                         ccol=c("red","orange","green"),
                         ...)
#TITLE draws a beta-binomial joint distribution
#DESCRIPTION
#   draws on the same plot, the marginal of the beta for \code{p}
#   in ordinates, the conditional distributions for \code{r|p}
#   and the marginal of \code{r}.
#DETAILS
#KEYWORDS 
#PKEYWORDS 
#INPUTS
#{n} <<Size of the binomial>>
#{a} <<First parameter of the beta distribution.>>
#{b} <<Second parameter of the beta distribution.>>
#[INPUTS]
#{pp} <<values of \code{p} to which draw the conditional of \code{r}.>>
#{rr} <<values of \code{r} to which draw the conditional of \code{p}.
#       These values are considered only when \code{pp} is \code{NULL}.>>
#{ci} <<coverage of a credibility interval to apply to all beta densities.>>
#{type} <<Indicates which type of distributions must be drawn. Choices are
#         "joint" or "conditional".>>
#{grafo} <<Indicates which kind of fitting must be done: either "fit" or
#           "complete". The last possibility means that all graphics will
#           be superimposable whatever is the choosen type.>>
#{mti} <<Main title for the plot.>>
#{sti} <<Sub title for the plot.>>
#{xlab} <<Label for the x-axis.>>
#{ylab} <<Label for the y-axis.>>
#{ccol} << confidence interval colors.>>
#{\dots} << to be given to the \code{plot} call.>>
#VALUE
# A complete plot is issued
#EXAMPLE
# draw8betabino(10,2,7);
#REFERENCE
#SEE ALSO
#CALLING
#COMMENT
#FUTURE
#AUTHOR J.-B. Denis
#CREATED 11_09_20
#REVISED 13_06_11
#--------------------------------------------
{
  ## checking
  check4tyle(type, "character",1, c("joint","conditional"),
             message = "'type' is not valid");
  check4tyle(grafo, "character",1, c("fit","complete"),
             message = "'grafo' is not valid");
  ## dealing with the marginal to draw
  if (type == "conditional") {
    if (is.null(pp)) {
      rmar <- TRUE; pmar <- FALSE;
      tistd <- "Conditional by beta";
      yti <- paste("For r =",paste(rr,collapse=" / "));
    } else {
      rmar <- FALSE; pmar <- TRUE;
      tistd <- "Conditional by binomial";
      yti <- paste("For p =",paste(pp,collapse=" / "));
    }
  } else {
    if (is.null(pp)) {
      tistd <- "Joint by beta";
      yti <- paste("For r =",paste(rr,collapse=" / "));
    } else {
      tistd <- "Joint by binomial";
      yti <- paste("For p =",paste(pp,collapse=" / "));
    }
    rmar <- TRUE; pmar <- TRUE;
  }
  ## dealing with the window to use
  if (grafo == "complete") {
    mg <- TRUE; mb <- TRUE;
  } else {
    if (type == "conditional") {
      if (is.null(pp)) {
        mg <- TRUE; md <- FALSE;
      } else {
        mg <- FALSE; md <- TRUE;
      }
    } else {
      mg <- TRUE; md <- TRUE;
    }
  }
  ## values to be considered
  val <- 0:n;
  ## computing the range for abscissaes
  ## of the r distributions
  xlim <- c(-1,1) + range(val);
  ## adding the room for the marginal beta
  xsep <- 1/12*diff(xlim);
  if (mg) {
    xlima <- xlim - c((1/3)*diff(xlim)+xsep,0);
  } else {
    xlima <- xlim;
  }
  if (is.null(pp)) {
    ## computing the parameters of the set of betas
    aa <- rr+a; bb <- n-rr+b;
    ## computing the heights for each beta
    rr <- sort(rr);
    hb <- rep(NA,length(rr));
    nb <- 123; xb <- seq(0,1,length=nb)[-c(1,nb)];
    for (ip in bf(rr)) {
      hb[ip] <- max(dbeta(xb,aa[ip],bb[ip]));
    }
    ### weighting it for the joint distribution
    if (type=="joint") {
      mama <- margibibe(n,a,b);
      hb <- hb * mama[rr+1];
    }
    ## computing the room for each beta
    ## (according to the rr asked values
    hp <- diff(c(rr,n+1));
    ## deducing the maximum scale
    masc <- max(hb/hp);
    ## computing the scale for each distribution
    echelle <- (hb/hp) / masc; 
    ## defining the positions for each density
    ou <- matrix(NA,length(rr),4);
    rra <- c(rr,n+1);
    for (ip in bf(rr)) {
      ou[ip,1:2] <- xlim[1] + c(rra[ip],rra[ip+1]) *
                              diff(xlim) / (n+1);
      ou[ip,3:4] <- c(0,1);
    }
  } else {
    ## computing the heights for each histogram
    pp <- sort(pp);
    hm <- rep(NA,length(pp));
    for (ip in bf(pp)) {
      hm[ip] <- max(dbinom(val,n,pp[ip]));
    }
    ### weighting it for the joint distribution
    if (type=="joint") {
      mama <- dbeta(pp,a,b);
      hm <- hm * mama;
    }
    ## computing the room for each histogram
    ## (according to the pp asked values
    hr <- diff(c(pp,1));
    ## deducing the maximum scale
    masc <- max(hm/hr);
    ## computing the scale for each histogram
    echelle <- (hm/hr) / masc; 
    ## defining the positions for each histo
    ou <- matrix(NA,length(pp),4);
    for (ip in bf(pp)) {
      ou[ip,1:2] <- xlim;
      ou[ip,3:4] <- c(pp,1)[c(ip,ip+1)];
    }
  }
  ## computing the range for ordinates
  plus <- 1/5;
  ylim <- c(-plus,1);
  ## adding room for the r marginal
  if (mb) {
    ylima <- ylim - c((1/5)*diff(ylim),0);
  } else {
    ylima <- xlim;
  }
  ## defining the box for the beta marginal
  oup <- c(xlima[1],xlim[1]-xsep,0,ylim[2]);
  ## defining the box for the r marginal
  our <- c(xlim,ylima[1],ylim[1]);
  ## preparing the labels
  lala <- unique(round(seq(0,n,le=7)));
  ## preparing the titles
  if (is.null(mti)) {
    mti <- tistd;
  }
  if (is.null(sti)) {
    sti <- paste("a =",a,"/ b =",b,"/ n =",n);
  }
  ## opening the graph
  if (is.null(xlab)) { xlab <- "Beta-Binomial density";}
  if (is.null(ylab)) { ylab <- yti;}
  plot(0,0,type="n",
       #yaxt="n",xaxt="n",
       xlab=xlab,
       ylab=ylab,
       xlim=xlima,ylim=ylima,
       main=mti,sub=sti,
       xaxt="n",yaxt="n",
       ...
      );
  ## writing the labels
  for (la in bf(lala)) {
    lal <- lala[la];
    lax <- 0.5 + xlim[1] + lal * diff(xlim) / (diff(range(val))+1);
    text(lax,0,lal,cex=2,pos=1);
  }
  if (is.null(pp)) {
    ## drawing each conditional beta
    for (ip in bf(rr)) {
      draw8beta(ou[ip,],aa[ip],bb[ip],pp=NULL,ci=ci,ccol=ccol[c(1,3)],
                echelle=echelle[ip],orientation="E");
      abline(h=ou[ip,2],lwd=1);
    }
  } else {
    ## drawing each conditional binomial
    for (ip in bf(pp)) {
      draw8bino(ou[ip,],n,pp[ip],ci=-1,echelle=echelle[ip],ccol=ccol);
      abline(h=ou[ip,2],lwd=1);
    }
  }
  ## drawing the marginal beta
  if (pmar) {
    draw8beta(oup,a,b,pp,ci=ci,orientation="W",ccol=ccol[c(1,3)]);
  }
  ## drawing the r marginal
  if (rmar) {
    prr <- margibibe(n,a,b);
    draw8histo(our,hist=list(n=0:n,f=prr),
              echelle=1,orientation="S",
              grafo=vector("list",0));
  }
  ## done
  invisible();
  }
#>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
